<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\PickupPointStoreRequest;
use App\Http\Requests\PickupPointUpdateRequest;
use App\Http\Resources\PickupPointResource;
use App\Models\PickupPoint;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PickupPointController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware('auth:sanctum');
        $this->middleware('demo')->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @param      \Illuminate\Http\Request  $request  The request
     *
     * @return     JsonResponse              The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $points = PickupPoint::latest()->get();
        return response()->json(PickupPointResource::collection($points));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param      PickupPointStoreRequest  $request  The request
     *
     * @return     JsonResponse             The json response.
     */
    public function store(PickupPointStoreRequest $request): JsonResponse
    {
        $pickupPoint = PickupPoint::create($request->validated());
        \App\Helpers\ActivityLogger::log(__('New pickup point added successfully'));

        return response()->json([
            'message' => __('Data saved successfully'),
            'point' => new PickupPointResource($pickupPoint),
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\PickupPoint  $pickupPoint  The pickup point
     *
     * @return     JsonResponse             The json response.
     */
    public function show(PickupPoint $pickupPoint): JsonResponse
    {
        return response()->json(new PickupPointResource($pickupPoint));

    }

    /**
     * Update the specified resource in storage.
     *
     * @param      PickupPointUpdateRequest  $request      The request
     * @param      \App\Models\PickupPoint   $pickupPoint  The pickup point
     *
     * @return     JsonResponse              The json response.
     */
    public function update(PickupPointUpdateRequest $request, PickupPoint $pickupPoint): JsonResponse
    {
        $pickupPoint->update($request->validated());
        \App\Helpers\ActivityLogger::log(__('Pickup point updated successfully'));
        return response()->json([
            'message' => __('Data updated successfully'),
        ]);
    }

    /**
     * Destroys the given pickup point.
     *
     * @param      \App\Models\PickupPoint  $pickupPoint  The pickup point
     *
     * @return     JsonResponse             The json response.
     */
    public function destroy(PickupPoint $pickupPoint): JsonResponse
    {
        $pickupPoint->delete();
        \App\Helpers\ActivityLogger::log(__('Pickup point removed successfully'));
        return response()->json([
            'message' => __('Data removed successfully'),
        ]);
    }
}
