<?php

namespace App\Models;

use EloquentFilter\Filterable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class GuestRoomReservation extends Model
{
    use Filterable;

    protected $fillable = [
        'uuid',
        'tracking',
        'payment_cleared',
        'note',
        'staff_note',
        'tax',
        'tax_amount',
        'total_room_service_charges',
        'payments',
        'booked_at',
        'checkin_at',
        'checkout_at',
        'reserved_at_date',
        'reserved_at_time',
        'reserved_till_date',
        'reserved_till_time',
        'reservation_status_id',
        'booking_source_id',
        'guest_room_id',
        'customer_id',
        'reservation_hours',
        'sub_total',
        'per_hour_charges'
    ];

    protected $casts = [
        'tax' => 'json',
        'payments' => 'json',
        'payment_cleared' => 'boolean'
    ];
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($reservation) {
            if (self::isOverlapping(
                $reservation->guest_room_id,
                $reservation->reserved_at_date,
                $reservation->reserved_at_time,
                $reservation->reserved_till_date,
                $reservation->reserved_till_time
            )) {
                throw new \Exception('The selected date and time duration is already reserved.');
            }
        });

        static::updating(function ($reservation) {
            if (self::isOverlapping(
                $reservation->guest_room_id,
                $reservation->reserved_at_date,
                $reservation->reserved_at_time,
                $reservation->reserved_till_date,
                $reservation->reserved_till_time,
                $reservation->id
            )) {
                throw new \Exception(__('The selected date and time duration is already reserved.'));
            }
        });
    }

    /**
     * Check if the given reservation period overlaps with existing reservations.
     *
     * @param  int  $guestRoomId
     * @param  string  $reservedAtDate
     * @param  string  $reservedAtTime
     * @param  string  $reservedTillDate
     * @param  string  $reservedTillTime
     * @param  int|null  $excludeReservationId
     * @return bool
     */
    public static function isOverlapping($guestRoomId, $reservedAtDate, $reservedAtTime, $reservedTillDate, $reservedTillTime, $excludeReservationId = null)
    {
        $query = self::where('guest_room_id', $guestRoomId)
            ->where(function ($query) use ($reservedAtDate, $reservedAtTime, $reservedTillDate, $reservedTillTime) {
                $query->where(function ($query) use ($reservedAtDate, $reservedAtTime, $reservedTillDate, $reservedTillTime) {
                    $query->where('reserved_at_date', '<=', $reservedTillDate)
                        ->where('reserved_till_date', '>=', $reservedAtDate)
                        ->where(function ($query) use ($reservedAtTime, $reservedTillTime) {
                            $query->where('reserved_at_time', '<=', $reservedTillTime)
                                ->where('reserved_till_time', '>=', $reservedAtTime);
                        });
                });
            });

        if ($excludeReservationId) {
            $query->where('id', '!=', $excludeReservationId);
        }

        return $query->exists();
    }


    public function scopeCheckedInReservations($query)
    {
        return $query->whereNotNull('checkin_at')->whereNull('checkout_at');
    }


    /**
     * Setting default route key
     *
     * @return string
     */
    public function getRouteKeyName(): string
    {
        return 'uuid';
    }


    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class, 'customer_id');
    }

    public function bookingSource(): BelongsTo
    {
        return $this->belongsTo(BookingSource::class, 'booking_source_id');
    }

    public function reservationStatus(): BelongsTo
    {
        return $this->belongsTo(ReservationStatus::class, 'reservation_status_id');
    }

    public function guestRoom(): BelongsTo
    {
        return $this->belongsTo(GuestRoom::class, 'guest_room_id');
    }

    public function sales(): HasMany
    {
        return $this->hasMany(Sale::class, 'guest_room_reservation_id');
    }


    public function getSalesItems()
    {
        return collect($this->sales)->sum('cart_total_items');
    }

    public function getSalesOrders()
    {
        return count($this->sales);
    }

    public function getSalesTaxAmount()
    {
        return collect($this->sales)->sum('tax_amount');
    }
    public function getSalesDiscountAmount()
    {
        return collect($this->sales)->sum('discount_amount');
    }
    public function getSalesPayableAmount()
    {
        return collect($this->sales)->sum('payable_after_all');
    }
}
