<?php

namespace App\Models;

use App\Channels\TwilioWhatsAppChannel;
use dacoto\EnvSet\Facades\EnvSet;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use NotificationChannels\Twilio\TwilioChannel;

class Setting extends Model
{
    protected $fillable = [
        'app_name',
        'app_address',
        'app_phone',
        'app_https',
        'app_url',
        'app_about',
        'app_date_format',
        'app_date_locale',
        'app_default_role',
        'app_background',
        'app_icon',
        'app_locale',
        'app_timezone', 'app_direction',
        'app_user_registration', 'mysqldump_path',

        'queue_connection',

        'mail_from_name',
        'mail_from_address',
        'mail_mailer',
        'mail_host',
        'mail_username',
        'mail_password',
        'mail_port',
        'mail_encryption',

        'mailgun_domain',
        'mailgun_endpoint',
        'mailgun_secret',

        'recaptcha_enabled',
        'recaptcha_public',
        'recaptcha_private',

        'currency_symbol',
        'currency_symbol_on_left',
        'tax_rate', 'is_vat',
        'is_tax_fix',
        'tax_id',
        'is_tax_included',

        'print_font_family',
        'print_name_address_position',
        'print_name_address_size',
        'print_header_footer_size',
        'print_items_font_size',
        'print_terms_conditions_font_size',
        'print_font_color',
        'printer_width',
        'printer_height',
        'signature_required',
        'terms_conditions',
        'print_auto',
        'invoice_type_title', 'display_logo',

        'tax_title',
        'advance_cashbook_mode',
        'advance_print_on_submit',
        'advance_customer_tip_collection',
        'advance_send_notification',

        'template_order_created',
        'template_order_cancelled',
        'template_order_checkout',

        'wa_status',
        'wa_twilio_account_sid',
        'wa_twilio_auth_token',
        'wa_twilio_from',
        'sms_status',
        'sms_channel',
        'twilio_account_sid', 'twilio_auth_token', 'twilio_from',
        'vonage_key', 'vonage_secret', 'vonage_from',
    ];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    protected static function boot(): void
    {
        parent::boot();

        /*
         * Register an updated model event with the dispatcher.
         *
         * @param \Closure|string $callback
         * @return void
         */
        self::updating(
            static function ($model) {
                $writeable = [
                    'app_url', 'app_name', 'app_https',
                    'app_timezone', 'app_locale', 'app_date_format',
                    'mail_from_address', 'mail_from_name',
                    'mail_mailer', 'mail_encryption',
                    'mail_host', 'mail_password',
                    'mail_port', 'mail_username',
                    'queue_connection',
                    'mailgun_domain', 'mailgun_secret',
                    'mailgun_endpoint', 'app_direction',
                    'advance_cashbook_mode',
                    'advance_print_on_submit',
                    'advance_customer_tip_collection',
                    'vonage_key', 'vonage_secret', 'vonage_from',
                    'twilio_account_sid', 'twilio_auth_token', 'twilio_from',
                    'wa_status',
                    'wa_twilio_account_sid',
                    'wa_twilio_auth_token',
                    'wa_twilio_from',
                    'mysqldump_path', 'display_logo'
                ];

                $writeable = collect($model)->only($writeable)->all();
                foreach ($writeable as $key => $value) {
                    EnvSet::setKey(strtoupper($key), $value);
                    EnvSet::save();
                }
            }
        );
    }

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'app_user_registration' => 'boolean',
        'recaptcha_enabled' => 'boolean',
        'app_https' => 'boolean',
        'sub_wid_is_flex' => 'boolean',
        'sub_wid_phone_input' => 'boolean',
        'sub_wid_email_input' => 'boolean',
        'currency_symbol_on_left' => 'boolean',
        'is_fix' => 'boolean',
        'is_vat' => 'boolean',
        'is_tax_fix' => 'boolean',
        'is_tax_included' => 'boolean',
        'signature_required' => 'boolean',
        'print_auto' => 'boolean',
        'sms_status' => 'boolean',
        'template_order_created' => 'json',
        'template_order_cancelled' => 'json',
        'template_order_checkout' => 'json',
        'wa_status' => 'boolean',
        'advance_send_notification' => 'boolean',
        'display_logo' => 'boolean',
    ];

    /**
     * Application icon URL
     *
     * @param mixed $icon icon
     *
     * @return string
     */
    public function getAppIconAttribute($icon): string
    {
        return $icon
            ? Storage::disk('public')->url($icon)
            : asset('images/default/icon.png');
    }

    /**
     * Application background image URL
     *
     * @param mixed $background background
     *
     * @return string
     */
    public function getAppBackgroundAttribute($background): string
    {
        return $background
            ? Storage::disk('public')->url($background)
            : asset('images/default/background.png');
    }


    public function getNotificationConfig($sale): array
    {
        $configArray = [];

        if ($sale->email) {
            array_push($configArray, 'mail');
        }

        if ($sale->phone && $this->wa_status) {
            array_push($configArray, TwilioWhatsAppChannel::class);
        }

        if ($sale->phone && $this->sms_status) {
            switch ($this->sms_channel) {
                case 'nexmo':
                    array_push($configArray, 'vonage');
                    break;
                case 'twilio':
                    array_push($configArray, TwilioChannel::class);
                    break;
                default:
                    // code...
                    break;
            }

            if (!$this->advance_send_notification) {
                $configArray = [];
            }
        }
        return [
            'mailer' => $configArray,
            'templates' => $this->getTemplates(),
        ];
    }


    protected function getTemplates()
    {
        return [
            'booked' => $this->template_order_created ?? $this->getDumyTemplat(),
            'cencelled' => $this->template_order_cancelled ?? $this->getDumyTemplat(),
            'checkout' => $this->template_order_checkout ?? $this->getDumyTemplat(),
        ];
    }

    protected function getDumyTemplat()
    {
        return [
            'mail_content' => 'Check at {order_link}',
            'sms_content' => 'Check at {tracking_link}',
            'subject' => ' # {tracking_id}',
        ];
    }
}
